
#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "PublicTypes.h"


#include "GlucoseMeter.h"
#include "SwTimer.h"
#include "SystemConfig.h"
#include "SerialCommands.h"
#include "Bpm.h"
#include "HWT.h"
#include "Spr.h"
#include "EcgDsc.h"
#include "PulseOximeter.h"
#include "OPAMP.h"

#include "virtual_com.h"
#include "usb_cdc.h"

#include "MCU.h"        /* MCU Specific */




#if DEBUG_MODE==FALSE
//#define SEND_SINE_WAVE

#define CONNECTED_TO_USB	1		//PTADD_PTADD0 == 1	//Change this
#define CONNECTED_TO_LCD_BOARD	0	//IRQSC_IRQF == 1
#define GUI_TOLD_ME_TO_GO_STAND_ALONE	0//	PTADD_PTADD2	//change this

/* unsecure flash */
const uint_8 sec@0x040f = 0x00; 
/* checksum bypass */
const uint_8 checksum_bypass@0x040a = 0x0; 


//const unsigned char boot@0xFFBA = 0x00; // bypass checksum 

typedef union 
{
	byte Byte;
	struct 
	{
		byte ConnectedToUsb      	:1;
		byte ConnectedToLcdBoard	:1;
		byte ZigbeeEnabled			:1;
		byte BluetoothEnabled		:1;
		byte :1;
		byte :1;
		byte :1;
		byte :1;
	} Bits;
} SystemFlags;


typedef enum
{
	NO_MEASUREMENT,
	GLU_MEASUREMENT,
	BPM_MEASUREMENT,
	BPM_LEAK_TEST,
	ECG_MEASUREMENT,
	SPR_MEASUREMENT,
	HEIGHT_MEASUREMENT,
	WEIGHT_MEASUREMENT,
	TEMPERATURE_MEASUREMENT
};



UINT8 InBuffer[32];	//From the PC to the MCU
UINT8 OutBuffer[128];	//From the MCU to the PC
UINT8 OutSize;
UINT8 InSize;


/* Receive Buffer */
extern uint_8 g_curr_recv_buf[DATA_BUFF_SIZE];
/* Send Buffer */
extern uint_8 g_curr_send_buf[DATA_BUFF_SIZE];
/* Receive Data Size */
extern uint_8 g_recv_size;
/* Send Data Size */
extern uint_8 g_send_size;



UINT16 IdNumber = 0;
UINT8 MainActualState;
UINT8 ActualMeasurement;

UINT8 TimerSendDummyData;

void TimerSendDummyData_Event(void);
void EnableAFE(UINT8 measurementId);


typedef enum
{
	MAIN_STATE_DISCONNECTED,
	MAIN_STATE_CONTROLLED_BY_LCD_BOARD,
	MAIN_STATE_CONTROLLED_BY_USB
} MainStates_e;

/* States */
void StateDisconnected(void);
void StateControlledByLcdBoard(void);
void StateControlledByUsb(void);

/*****************************************************************************
 * Local Functions Prototypes
 *****************************************************************************/
static void Init_Sys(void);
static void Mcu_Init(void);
static void MCG_Init(void);

//void CDC_Engine(void);




/* AFEs functions */

/* GlucoseMeter */
void GluStartMeasurementReq(void);
void GluAbortMeasurementReq(void);
void GluStartCalibrationReq(void);
void GlucometerBloodDetectedInd(void);
void GlucometerMeasurementOkInd(void);
void GlucometerCalibrationOkInd(void);
void GlucometerDebugNewDataReadyInd(void);


/* Bpm */
void BpmStartMeasurementReq(void);
void BpmAbortMeasurementReq(void);
void BpmMeasurementCompleteInd(void);
void BpmMeasurementErrorInd(void);
void BpmNewDataReadyInd(void);	//for debug only
void BmpSendPressureValue(void);


void BpmStartLeakTestReq(void);
void BpmAbortLeakTestReq(void);
void BpmLeakTestCompleteInd(void);
void BpmCuffDeinflated(void);

/* Ecg */
void EcgHeartRateStartMeasurementReq(void);
void EcgHeartRateAbortMeasurementReq(void);
void EcgHeartRateMeasurementCompleteOkInd(void);
void EcgHeartRateMeasurementErrorInd(void);
void EcgHeartBeatOccurredInd(void);

void EcgDiagnosticModeStartMeasurementReq(void);
void EcgDiagnosticModeStopMeasurementReq(void);
void EcgDiagnosticModeNewDataReadyInd(void);

/* Hwt */
void HwtReadTemperatureReq(void);
void HwtReadHeightReq(void);
void HwtReadWeightReq(void);
void HwtTemperatureMeasurmentCompleteInd(void);
void HwtHeightMeasurementCompleteInd(void);
void HwtWeightMeasurementCompleteInd(void);


/* Spr */
void SprDiagnosticModeStartMeasurementReq(void);
void SprDiagnosticModeStopMeasurementReq(void);
void SprDiagnosticModeNewDataReadyInd(void);


/* Pox */
void PoxStartMeasurementReq(void);
void PoxAbortMeasurementReq(void);
void PoxMeasurementCompleteOkInd(void);
void PoxMeasurementErrorInd(void);


void PoxDiagnosticModeStartMeasurementReq(void);
void PoxDiagnosticModeStopMeasurementReq(void);
void PoxNewDataReadyInd(void);
void PoxDebugModeNewDataReadyInd(void);

/* system restart */
void SystemRestartReq(void);

/* Spr 2nd pass */
const pFunc_t Spr_Events[] = 
{
	NULL,//EVENT_SPR_NONE,
	NULL,//EVENT_SPR_MEASUREMENT_COMPLETE_OK,
	NULL,//EVENT_SPR_MEASUREMENT_ERROR,
	SprDiagnosticModeNewDataReadyInd	//EVENT_SPR_NEW_DATA_READY
};


/* Serial comm */
void SerialComm_PeriodicTask(void);
void SerialComm_SendData(void);



/** Pointer to function that will be used as a callback */

const pFunc_t GlucoseMeter_Events[] = 
{
	NULL,						//EVENT_GLU_NONE,
	GlucometerBloodDetectedInd,	//EVENT_GLU_BLOOD_DETECTED,
	GlucometerMeasurementOkInd,	//EVENT_GLU_FINISH_MEASUREMENT_OK,
	GlucometerCalibrationOkInd,	//EVENT_GLU_FINISH_CALIBRATION_OK
	GlucometerDebugNewDataReadyInd,	//EVENT_GLU_DEBUG_NEW_DATA_READY
};


const pFunc_t EcgDsc_Events[] = 
{
	NULL,									//EVENT_ECG_NONE,
	EcgHeartRateMeasurementCompleteOkInd,	//EVENT_ECG_HEART_RATE_MEASUREMENT_COMPLETE_OK,
	EcgHeartRateMeasurementErrorInd,		//EVENT_ECG_HEART_RATE_MEASUREMENT_ERROR,
	EcgHeartBeatOccurredInd,				//EVENT_ECG_HEART_BEAT_OCCURRED,
	EcgDiagnosticModeNewDataReadyInd		//EVENT_ECG_DIAGNOSTIC_MODE_NEW_DATA_READY	
};

const pFunc_t Ecg_Events[] = 
{
	NULL,									//EVENT_ECG_NONE,
	EcgHeartRateMeasurementCompleteOkInd,	//EVENT_ECG_HEART_RATE_MEASUREMENT_COMPLETE_OK,
	EcgHeartRateMeasurementErrorInd,		//EVENT_ECG_HEART_RATE_MEASUREMENT_ERROR,
	NULL,	//EcgHeartBeatOccurredInd,				//EVENT_ECG_HEART_BEAT_OCCURRED,
	EcgDiagnosticModeNewDataReadyInd		//EVENT_ECG_DIAGNOSTIC_MODE_NEW_DATA_READY	
};


const pFunc_t Bpm_Events[] = 
{
	NULL,						//EVENT_BPM_NONE,
	BpmMeasurementCompleteInd,	//EVENT_BPM_MEASUREMENT_COMPLETE_OK,
	BpmMeasurementErrorInd,		//EVENT_BPM_MEASUREMENT_ERROR,
	BpmLeakTestCompleteInd,		//EVENT_BPM_LEAK_TEST_COMPLETE,
	BpmNewDataReadyInd,			//EVENT_BPM_NEW_DATA_READY	//for debug only
	BmpSendPressureValue,		//EVENT_BPM_SEND_PRESSURE_VALUE_TO_PC
  BpmCuffDeinflated			//EVENT_BPM_CUFF_DEINFLATED
};


const pFunc_t Pox_Events[] = 
{
	NULL,							//POX_EVENT_NONE,
	PoxMeasurementCompleteOkInd, 	//POX_EVENT_MEASUREMENT_COMPLETE_OK,
	PoxMeasurementErrorInd,			//POX_EVENT_MEASUREMENT_ERROR,
	PoxNewDataReadyInd,				//POX_EVENT_NEW_DATA_READY
	PoxDebugModeNewDataReadyInd		//POX_EVENT_DEBUG_MODE_NEW_DATA_READY
};


const pFunc_t Hwt_Events[] = 
{
	NULL,		//EVENT_HWT_NONE,	
	HwtHeightMeasurementCompleteInd,		//EVENT_HEIGHT_MEASUREMENT_ERROR, //Send the height withh value of 0
	HwtHeightMeasurementCompleteInd,		//EVENT_HEIGHT_MEASUREMENT_COMPLETE_OK,
	HwtWeightMeasurementCompleteInd,		//EVENT_WEIGHT_MEASUREMENT_COMPLETE_OK,
	HwtTemperatureMeasurmentCompleteInd		//EVENT_TEMPERATURE_MEASUREMENT_COMPLETE_OK		
};


const pFunc_t MainStateMachine[] = 
{
	StateDisconnected,
	StateControlledByLcdBoard,
	StateControlledByUsb
};



const pFunc_t ExecuteCommandReq[] =
{
	GluStartMeasurementReq,				//GLU_START_MEASUREMENT,
	GluAbortMeasurementReq,				//GLU_ABORT_MEASUREMENT,
	GluStartCalibrationReq,				//GLU_START_CALIBRATION,
	NULL,								//GLU_BLOOD_DETECTED,
	NULL,								//GLU_MEASUREMENT_COMPLETE_OK,
	NULL,								//GLU_CALIBRATION_COMPLETE_OK,
	
	BpmStartMeasurementReq,				//BPM_START_MEASUREMENT,
	BpmAbortMeasurementReq,				//BPM_ABORT_MEASUREMENT,
	NULL,								//BPM_MEASUREMENT_COMPLETE_OK,
	NULL,								//BPM_MEASUREMENT_ERROR,
										  
	BpmStartLeakTestReq,				//BPM_START_LEAK_TEST,
	BpmAbortLeakTestReq,					//BPM_ABORT_LEAK_TEST,
	NULL,								//BPM_LEAK_TEST_COMPLETE,	
	
	EcgHeartRateStartMeasurementReq,	//ECG_HEART_RATE_START_MEASUREMENT,
	EcgHeartRateAbortMeasurementReq,	//ECG_HEART_RATE_ABORT_MEASUREMENT,
	NULL,								//ECG_HEART_RATE_MEASUREMENT_COMPLETE_OK,
	NULL,								//ECG_HEART_RATE_MEASUREMENT_ERROR,
	NULL,								//ECG_HEART_BEAT_OCCURRED,
	
	EcgDiagnosticModeStartMeasurementReq,//ECG_DIAGNOSTIC_MODE_START_MEASUREMENT,
	EcgDiagnosticModeStopMeasurementReq, //ECG_DIAGNOSTIC_MODE_STOP_MEASUREMENT,
	NULL,								 //ECG_DIAGNOSTIC_MODE_NEW_DATA_READY,	
		
	HwtReadTemperatureReq,				//TMP_READ_TEMPERATURE,
	HwtReadHeightReq,					//HGT_READ_HEIGHT,
	HwtReadWeightReq,					//WGT_READ_WEIGHT,

	NULL,	//SprStartMeasurementReq,				//SPR_START_MEASUREMENT,
	NULL,	//SprAbortMeasurementReq,				//SPR_ABORT_MEASUREMENT,
	NULL,								//SPR_MEASUREMENT_COMPLETE_OK,
	NULL,								//SPR_MEASUREMENT_ERROR,

	SprDiagnosticModeStartMeasurementReq,								//SPR_DIAGNOSTIC_MODE_START_MEASURMENT,
	SprDiagnosticModeStopMeasurementReq,								//SPR_DIAGNOSTIC_MODE_STOP_MEASURMENT,
	NULL,								//SPR_DIAGNOSTIC_MODE_NEW_DATA_READY,
	NULL,								//SPR_DIAGNOSTIC_MODE_MEASUREMENT_COMPLETE_OK,
	NULL,								//SPR_DIAGNOSTIC_MODE_MEASUREMENT_ERROR,
		
	PoxStartMeasurementReq,				//POX_START_MEASUREMENT,
	PoxAbortMeasurementReq,				//POX_ABORT_MEASURMENT,
	NULL,								//POX_MEASURMENT_COMPLETE_OK,
	NULL,								//POX_MEASURMENT_ERROR,

	PoxDiagnosticModeStartMeasurementReq,//POX_DIAGNOSTIC_MODE_START_MEASURMENT,
	PoxDiagnosticModeStopMeasurementReq,//POX_DIAGNOSTIC_MODE_STOP_MEASURMENT,
	NULL,								//POX_DIAGNOSTIC_MODE_NEW_DATA_READY
	NULL,	//BPM_SEND_PRESSURE_VALUE_TO_PC,
	SystemRestartReq,	//SYSTEM_RESTART,
	NULL	//BPM_DATA_READY = 0xFF,
};



void main(void) 
{
    
  Init_Sys();
    
  ///*MCU_Init();    

	AFE_ENABLE_PIN_INIT();
  //AFE_ENABLE();		//power on board
	
		
  //opamp_gp_mode();
	//TRIAMP1C0_HighMode();
	//TRIAMP2C0_HighMode(); 

	SwTimer_Init();      

	//Bpm_Init();	
	//need to turn on AFE before init SPR
	//Spr_Init();
	///*Pox_Init();
	///*EcgDsc_Init();	
	GlucoseMeter_Init();	
	//Hwt_Init();
	EnableInterrupts;
  
  ///*
  (void)TestApp_Init(); /* Initialize the USB Test Application */  
 
			
	
	TimerSendDummyData = SwTimer_CreateTimer(TimerSendDummyData_Event);
	
		
	for(;;) 
	{
		GlucoseMeter_PeriodicTask();
		///*EcgDsc_PeriodicTask();
		//Bpm_PeriodicTask();		
	  //Hwt_PeriodicTask();
		///*Spr_PeriodicTask();
		///*Pox_PeriodicTask();
		
		
		SwTimer_PeriodicTask();
		SerialComm_PeriodicTask();
		MainStateMachine[MainActualState]();
		
	  
		
		__RESET_WATCHDOG(); /* feeds the dog */
	} /* loop forever */
}


static const UINT16 SinX[] = 
{
	1250, 1349, 1448, 1545, 1640, 1733, 1823, 1910, 1992, 2069, 2142, 2208, 2269, 2323, 2371, 2411, 2444, 2470, 2488, 2498, 2500, 2494, 2481, 2459, 2430, 2394, 2350, 2300, 2243, 2179, 2110, 2036, 1956, 1872, 1784, 1693, 1598, 1502, 1404, 1305, 1206, 1107, 1009, 912, 818, 726, 638, 553, 473, 398, 328, 264, 206, 155, 111, 73, 44, 21, 7, 0, 2, 11, 28, 53, 85, 124, 171, 225, 284, 350, 422, 499, 581, 667, 756, 849, 944, 1041, 1140, 1239
};

#define SIN_X_LAST_ELEMENT 	79
volatile UINT8 EcgBufferSize = 64;
volatile UINT8 Time	= 64;


void TimerSendDummyData_Event(void)
{
	if (ActualMeasurement == ECG_MEASUREMENT)
	{		
		static UINT8 SinXArrayActualElement = 0;

		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = ECG_DIAGNOSTIC_MODE_NEW_DATA_READY;
		OutBuffer[OutSize++] = EcgBufferSize + 2;				//data bytes
		
		OutBuffer[OutSize++] = (UINT8) (IdNumber >> 8);
		OutBuffer[OutSize++] = (UINT8) (IdNumber & 0x00FF);
		
		IdNumber++;
		
		
		while (OutSize < (EcgBufferSize + DATA_PACKET + 2))
		{
			OutBuffer[OutSize++] = SinX[SinXArrayActualElement] >> 8;		
			OutBuffer[OutSize++] = SinX[SinXArrayActualElement] & 0x00FF;
			
			if (SinXArrayActualElement == SIN_X_LAST_ELEMENT)
			{
				SinXArrayActualElement = 0;
			}
			else
			{
				SinXArrayActualElement++;
			}
		}

		(void)SerialComm_SendData();
		
		SwTimer_StartTimer(TimerSendDummyData, Time);
	}	
	
}

void StateDisconnected(void)
{
	if (CONNECTED_TO_USB)
	{
		MainActualState = MAIN_STATE_CONTROLLED_BY_USB;
	}
		else if (CONNECTED_TO_LCD_BOARD)
	{
		MainActualState = MAIN_STATE_CONTROLLED_BY_LCD_BOARD;
	}	
}


void StateControlledByLcdBoard(void)
{
	if (CONNECTED_TO_USB)
	{
		//Send LL16 to sleep
		//add code here
		MainActualState = MAIN_STATE_CONTROLLED_BY_USB;
	}
	else if (!CONNECTED_TO_LCD_BOARD)
	{
		MainActualState = MAIN_STATE_DISCONNECTED;
	}
	
	//run SPI process here for communication with LL16 
}



void StateControlledByUsb(void)
{
	if (CONNECTED_TO_LCD_BOARD)
	{
		if (!CONNECTED_TO_USB || GUI_TOLD_ME_TO_GO_STAND_ALONE)
		{
			//wake up LL16
			//add code here
			MainActualState = MAIN_STATE_CONTROLLED_BY_LCD_BOARD;
		}
	}
	else //!CONNECTED_TO_LCD_BOARD
	{
		if (!CONNECTED_TO_USB)
		{
			MainActualState = MAIN_STATE_DISCONNECTED;	
		}		
	}
		
	///*CDC_Engine();	//run CDC task
}









void SerialComm_PeriodicTask(void)
{	
 	if (MainActualState == MAIN_STATE_CONTROLLED_BY_USB)
	{
   				
		static uint_8 status = 0;///*
		//if there is data, read it
		if(g_recv_size)   //*
	  {
	    ///*InSize = CDC_GET_DATA(InBuffer);	
	    
	    /* Copy Received Buffer to InBuffer   
        for (status = 0; status < g_recv_size; status++)
        {
            InBuffer[status] = g_curr_recv_buf[status];
        }
         
       */  
        InSize = g_recv_size;
	    
	    
	    
		}
	}
	else if (MainActualState == MAIN_STATE_CONTROLLED_BY_LCD_BOARD)
	{
		//check if there is new data from the LCD board
		//add SPI receive function here	
	}


	//if there is data, parse it and execute command
	if (InSize>0)   ///*   g_recv_size
	{	
		//there is new data
		if (g_curr_recv_buf[PACKET_TYPE] == REQ)
		{
			if (ExecuteCommandReq[g_curr_recv_buf[COMMAND_OPCODE]] != NULL)
			{
				//check if OPCCODE is in a valid range
				if ((g_curr_recv_buf[COMMAND_OPCODE] <= LAST_COMMAND))
				{
					ExecuteCommandReq[g_curr_recv_buf[COMMAND_OPCODE]]();
				}
			}	
		}
		//else
		//{
			//packet type is not a request
		//}
		
		g_recv_size=0;
		InSize = 0;
	}
}

    





void SerialComm_SendData(void)
{
	  
	if (MainActualState == MAIN_STATE_CONTROLLED_BY_USB)
	{	
			
		if (OutSize>0)			 ///*  OutSize>0
		{			
			//TO DO:
			//check if previous data has been sent
			///*SEND_DATA_TO_HOST(OutBuffer, OutSize);
			
			USB_Class_CDC_Interface_DIC_Send_Data(CONTROLLER_ID,OutBuffer,OutSize);
      OutSize = 0;
		}
	}
	else if (MainActualState == MAIN_STATE_CONTROLLED_BY_LCD_BOARD)
	{
		//add SPI send function here	

	}	
}





#define MEDICAL_SENSOR_BOARD_IIC_ADDRESS		0x03

const UINT8 CmdEnableGlu	=	0x11;	
const UINT8 CmdEnableSpr	=	0x22;	
const UINT8 CmdEnablePox	=	0x33;	
const UINT8 CmdEnableEcg	=	0x44;	
const UINT8 CmdEnableBpm	=	0x55;	
const UINT8 CmdEnableHwt	=	0x66;	
const UINT8 CmdDisableAll	=	0xFF;



void EnableAFE(UINT8 measurementId)
{
	UINT8 delay=0xFF;
	
	AFE_DISABLE();		//power off board
	
	while(delay--)	//wait a few us
	{
		;
	}
	
	IIC_DISABLE();
	vfnIICV1Init();
	AFE_ENABLE();		//power on board
	
	switch (measurementId)		//send IIC data to select device
	{
		case NO_MEASUREMENT:
			break;
			
		case GLU_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, (UINT8*)&CmdEnableGlu, 1);
			break;
			
		case BPM_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableBpm, 1);
			break;
						
		case ECG_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableEcg, 1);
			break;
		
		case SPR_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableSpr, 1);
			break;
			
		case HEIGHT_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableHwt, 1);
			break;		
		
		case WEIGHT_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableHwt, 1);
			break;		
		
		case TEMPERATURE_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, &CmdEnableHwt, 1);
			break;		
	}	
}


void SystemRestartReq(void)
{
	//stop all measurements
	GlucoseMeter_AbortMeasurement();
	EcgDsc_DiagnosticModeStopMeasurement();
	Bpm_AbortMeasurement();
	Bpm_AbortLeakTest();
	Spr_AbortMeasurement();
	
	Hwt_StopReadingTemperature();	
	Hwt_StopReadingHeight();
	Hwt_StopReadingWeight();
}

/**************************************************
		Glucose Meter
******************************************************/

/**
	Starts a glucose measurment
*/
void GluStartMeasurementReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{
		AFE_ENABLE();
		
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = GLU_START_MEASUREMENT;
		OutBuffer[OutSize++] = 1;	//data bytes

		if (GlucoseMeter_StartMeasurement() == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}

		(void)SerialComm_SendData();
		ActualMeasurement = GLU_MEASUREMENT;	
	}
}


/**
	Aborts a glucose measurement
*/
void GluAbortMeasurementReq(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{		
		//execute command and send confirm
		GlucoseMeter_AbortMeasurement();
		
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = GLU_ABORT_MEASUREMENT;
		OutBuffer[OutSize++] = 0;	//data bytes
				
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
		
		AFE_DISABLE();
	}
}


/**
	Starts a calibrarion routine in the glucose meter
*/
void GluStartCalibrationReq(void)
{
	UINT16 knownGlucoseValue;

	if (ActualMeasurement == NO_MEASUREMENT)
	{
		AFE_ENABLE();
		
		knownGlucoseValue = InBuffer[DATA_PACKET]<<8;
		knownGlucoseValue += InBuffer[DATA_PACKET+1];

		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = GLU_START_CALIBRATION;
		OutBuffer[OutSize++] = 1;	//data bytes

		if (GlucoseMeter_StartCalibration(knownGlucoseValue) == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}
				
		(void)SerialComm_SendData();
		ActualMeasurement = GLU_MEASUREMENT;
	}
}

/**
	Sends an IND telling that the user has placed blood in the strip
*/
void GlucometerBloodDetectedInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{		
		//Send GLU_BLOOD_DETECTED indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_BLOOD_DETECTED;
		OutBuffer[OutSize++] = 0;				//data bytes
		(void)SerialComm_SendData();
	}			
}


/**
	Sends an IND telling that the glucose test has finished.
	It sends to the host the glucose value measured.
*/
void GlucometerMeasurementOkInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{		
		//Send  indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_MEASUREMENT_COMPLETE_OK;
		OutBuffer[OutSize++] = 2;	//data bytes
		OutBuffer[OutSize++] = GlucoseMeter_GlucoseValue >> 8;			//high byte
		OutBuffer[OutSize++] = GlucoseMeter_GlucoseValue & 0x00FF;		//low byte
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
		
		AFE_DISABLE();
	}
}


/**
	Sends an IND telling that the calibrarion routine finished correctly.
*/
void GlucometerCalibrationOkInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{
		//Send  indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_CALIBRATION_COMPLETE_OK;
		OutBuffer[OutSize++] = 2;	//data bytes: calibration constant
		OutBuffer[OutSize++] = (UINT8)(GlucoseMeter_CalibrationConstant >> 8);		//high byte
		OutBuffer[OutSize++] = (UINT8)(GlucoseMeter_CalibrationConstant & 0x00FF);	//low byte
		(void)SerialComm_SendData();		
		ActualMeasurement = NO_MEASUREMENT;
		
		AFE_DISABLE();
	}
}

/**
	This function is only used during debug to send the actual ADC value read by the MCU.
	This function is activated if the GLU_DEBUG is defined in GlucoseMeter.h
*/
void GlucometerDebugNewDataReadyInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{
		UINT8 i;
		
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_DEBUG_MODE_NEW_DATA_READY;
		OutBuffer[OutSize++] = GLU_GRAPH_DATA_ARRAY_SIZE;	//data bytes: real time graph
		
		
		for (i=0; i<GLU_GRAPH_DATA_ARRAY_SIZE; i++)
		{			
			OutBuffer[OutSize++] = GlucoseMeter_GraphDataToPc[i];
		}
		(void)SerialComm_SendData();		
	}
	
}


/*********************************
		Ecg
**********************************/		



void EcgHeartRateStartMeasurementReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{		
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = ECG_HEART_RATE_START_MEASUREMENT;
		OutBuffer[OutSize++] = 1;	//data bytes
		
		EnableAFE(ECG_MEASUREMENT);
		
	#ifdef DSC_BOARD_CONNECTED
		if (EcgDsc_HeartRateStartMeasurement() == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}
		
		(void)SerialComm_SendData();		


	#else
		if (Ecg_HeartRateStartMeasurement() == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}
		
		(void)SerialComm_SendData();		
	#endif	
		ActualMeasurement = ECG_MEASUREMENT;
	}
}


void EcgHeartRateAbortMeasurementReq(void)
{
	if (ActualMeasurement == ECG_MEASUREMENT)
	{		
		EcgDsc_DiagnosticModeStopMeasurement();
		
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = ECG_HEART_RATE_ABORT_MEASUREMENT;
		OutBuffer[OutSize++] = 0;	//data bytes


		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;			
		
		AFE_DISABLE();
	}
}



void EcgHeartRateMeasurementErrorInd(void)
{
	if (ActualMeasurement == ECG_MEASUREMENT)
	{		
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = ECG_HEART_RATE_MEASUREMENT_ERROR;
		OutBuffer[OutSize++] = 0;
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
		
		AFE_DISABLE();
	}
}


UINT8 Ecg_HeartRate = 0;

void EcgHeartRateMeasurementCompleteOkInd(void)
{
	if (ActualMeasurement == ECG_MEASUREMENT)
	{
		//Send  indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = ECG_HEART_RATE_MEASUREMENT_COMPLETE_OK;
		OutBuffer[OutSize++] = 1;	//data bytes
		OutBuffer[OutSize++] = Ecg_HeartRate;               
		(void)SerialComm_SendData();	
		ActualMeasurement = NO_MEASUREMENT;
		AFE_DISABLE();
	}
}


void EcgHeartBeatOccurredInd(void)
{
	
}


/*
	Ecg Dagnostic mode
*/

void EcgDiagnosticModeStartMeasurementReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = ECG_DIAGNOSTIC_MODE_START_MEASUREMENT;
		OutBuffer[OutSize++] = 1;	//data bytes

#ifdef SEND_SINE_WAVE
		OutBuffer[OutSize++] = ERROR_OK;
		SwTimer_StartTimer(TimerSendDummyData, 20);

#else
		EnableAFE(ECG_MEASUREMENT);
		if (EcgDsc_DiagnosticModeStartMeasurement() == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}					
#endif

		(void)SerialComm_SendData();			
		
		ActualMeasurement = ECG_MEASUREMENT;
	}
}


void EcgDiagnosticModeStopMeasurementReq(void)
{
	if (ActualMeasurement == ECG_MEASUREMENT)
	{			

#ifdef SEND_SINE_WAVE
		SwTimer_StopTimer(TimerSendDummyData);		
#else
		EcgDsc_DiagnosticModeStopMeasurement();
#endif
		
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = ECG_DIAGNOSTIC_MODE_STOP_MEASUREMENT;
		OutBuffer[OutSize++] = 0;	//data bytes
				
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
		AFE_DISABLE();
	}
}





void EcgDiagnosticModeNewDataReadyInd(void)
{
	if (ActualMeasurement == ECG_MEASUREMENT)
	{		
		UINT8 i=0;
		
		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = ECG_DIAGNOSTIC_MODE_NEW_DATA_READY;
		OutBuffer[OutSize++] = DATA_LENGTH_FROM_DSC;				//data from DSC + packetID
		
		OutBuffer[OutSize++] = (UINT8) (IdNumber >> 8);
		OutBuffer[OutSize++] = (UINT8) (IdNumber & 0x00FF);
		
		IdNumber++;
				
		//copy data from DSC to outbuffer
		i = DATA_START_POSITION;
		
		while (i<DATA_END_POSITION)
		{
			OutBuffer[OutSize++] = DataFromDsc[i+1];	//swap data bytes
			OutBuffer[OutSize++] = DataFromDsc[i];
			i+=2;
		}
		
		//send data
		(void)SerialComm_SendData();
	}	
}






/***************************************
		Bpm
****************************************/		

/**
	Starts a Blood Pressure measurement and sends a CFM
*/

void BpmStartMeasurementReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{	
	  AFE_ENABLE();  //EnableAFE(BPM_MEASUREMENT);   ///*
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = BPM_START_MEASUREMENT;
		OutBuffer[OutSize++] = 1;	//data bytes

		
		if (Bpm_StartMeasurement() == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}		

		(void)SerialComm_SendData();	
		ActualMeasurement = BPM_MEASUREMENT;
	}
}

 /**
	Stops the actual Blood Pressure measurement and sends a CFM
*/
void BpmAbortMeasurementReq(void)
{
	if (ActualMeasurement == BPM_MEASUREMENT)
	{		
		Bpm_AbortMeasurement();
			
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = BPM_ABORT_MEASUREMENT;
		OutBuffer[OutSize++] = 1;	//data bytes
		(void)SerialComm_SendData();	
		ActualMeasurement = NO_MEASUREMENT;
		//BPM_OPEN_VALVE();
	//	AFE_DISABLE();  ///*
	}
}



/**
	This IND is generated when the BPM test has been completed.
	It sends the following bytes to the host:
		- Systolic pressure
		- Diastolic pressure
		- Heart rate
		- Mean arterial pressure
*/

void BpmMeasurementCompleteInd(void)
{
	if (ActualMeasurement == BPM_MEASUREMENT)
	{		
		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = BPM_MEASUREMENT_COMPLETE_OK;
		OutBuffer[OutSize++] = 4;									//data bytes
		
		
		OutBuffer[OutSize++] = Bpm_SystolicPressure;
		OutBuffer[OutSize++] = Bpm_DiastolicPressure;
		OutBuffer[OutSize++] = (UINT8)Bpm_HeartRate;
		OutBuffer[OutSize++] = Bpm_MeanArterialPressure;
		
		//data is alredy in the buffer
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
	//	BPM_OPEN_VALVE();
	//	AFE_DISABLE();  ///*
	}
}

 /**
	This IND is sent by the MCU to the host when the blood pressure measurment 
	fails for some reason. The error code is sent using 1 byte.
*/

void BpmMeasurementErrorInd(void)
{
	if (ActualMeasurement == BPM_MEASUREMENT)
	{
		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = BPM_MEASUREMENT_ERROR;
		OutBuffer[OutSize++] = 1;				//data bytes
		
		OutBuffer[OutSize++] = Bpm_ErrorCode;	//Send error code ///*JM
				
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
	//	BPM_OPEN_VALVE();
	//	AFE_DISABLE();  ///*
	}
}

/**
	This IND is only sent if BPM_DEBUG is enabled. It will send the actual ADC readings
	for debugging purposes.
*/

void BpmNewDataReadyInd(void)
{
	#ifdef BPM_DEBUG
	UINT8 i;
	if ((ActualMeasurement == BPM_MEASUREMENT) || (ActualMeasurement == BPM_LEAK_TEST))
	{		
		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = BPM_DEBUG_MODE_NEW_DATA_READY;
		OutBuffer[OutSize++] = BPM_BUFFER_SIZE;				//data bytes
		///*JM OutSize+=BPM_BUFFER_SIZE;
		
		for (i=0; i<BPM_BUFFER_SIZE; i++)
		{
			OutBuffer[OutSize++] = Bpm_GraphData[i];
		}
		//data is alredy in the buffer
		(void)SerialComm_SendData();
	}
  #endif
}

 /**
	This is generated periodically by the MCU when the cuff is being inflated.
	It sends the acutal pressure value to the host using an IND.
*/
void BmpSendPressureValue(void)
{
	if (ActualMeasurement == BPM_MEASUREMENT)
	{	
		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = BPM_SEND_PRESSURE_VALUE_TO_PC;
		OutBuffer[OutSize++] = 1;				//data bytes
		OutBuffer[OutSize++] = Bpm_ActualPressureValue;
				
		//data is alredy in the buffer
		(void)SerialComm_SendData();	
	}
}

/**
	Starts a leak test and sends CFM.	
*/

void BpmStartLeakTestReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{		
		AFE_ENABLE();  //EnableAFE(BPM_MEASUREMENT); ///*
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = BPM_START_LEAK_TEST;
		OutBuffer[OutSize++] = 1;	//data bytes

		
		if (Bpm_StartLeakTest() == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}		
		
		(void)SerialComm_SendData();
		
		ActualMeasurement = BPM_LEAK_TEST;	
	}
}

/**
	Aborts a leak test and sends CFM
*/
void BpmAbortLeakTestReq(void)
{
	if (ActualMeasurement == BPM_LEAK_TEST)
	{
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = BPM_ABORT_LEAK_TEST;
		OutBuffer[OutSize++] = 0;	//data bytes

		Bpm_AbortLeakTest();
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
	//	BPM_OPEN_VALVE();
	//	AFE_DISABLE();  ///*
	}
}
/**
	Sends an IND telling the host the leak test result
*/
void BpmLeakTestCompleteInd(void)
{
	if (ActualMeasurement == BPM_LEAK_TEST)
	{
		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = BPM_LEAK_TEST_COMPLETE;
		OutBuffer[OutSize++] = 1;				//data bytes
			
		OutBuffer[OutSize++] = Bpm_LeakTestResult;				//data bytes
		
		//data is alredy in the buffer
		(void)SerialComm_SendData();	
		ActualMeasurement = NO_MEASUREMENT;
		//BPM_OPEN_VALVE();
		//AFE_DISABLE();  ///*
	}
}

/**
	Event generated by BPM module indicating that the cuff has been de inflated
*/

void BpmCuffDeinflated(void)
{
	AFE_DISABLE();
}

  

/* Tpm */
void HwtReadTemperatureReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{		
		Hwt_StartReadingTemperature();
		ActualMeasurement = TEMPERATURE_MEASUREMENT;
	}
}


void HwtReadHeightReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{		
		Hwt_StartReadingHeight();	
		ActualMeasurement = HEIGHT_MEASUREMENT;
	}
}

void HwtReadWeightReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{
		Hwt_StartReadingWeight();
		ActualMeasurement = WEIGHT_MEASUREMENT;
	}
}



void HwtTemperatureMeasurmentCompleteInd(void)
{
	OutBuffer[OutSize++] = CFM;
	OutBuffer[OutSize++] = TMP_READ_TEMPERATURE;
	OutBuffer[OutSize++] = 2;	//data bytes

	OutBuffer[OutSize++] = Hwt_Temperature >> 8;			//high byte
	OutBuffer[OutSize++] = Hwt_Temperature & 0x00FF;		//low byte

	(void)SerialComm_SendData();	
	ActualMeasurement = NO_MEASUREMENT;			
}


void HwtHeightMeasurementCompleteInd(void)
{
	OutBuffer[OutSize++] = CFM;
	OutBuffer[OutSize++] = HGT_READ_HEIGHT;
	OutBuffer[OutSize++] = 1;	//data bytes

	OutBuffer[OutSize++] = Hwt_Height;	
	(void)SerialComm_SendData();
	ActualMeasurement = NO_MEASUREMENT;					
}


void HwtWeightMeasurementCompleteInd(void)
{
	OutBuffer[OutSize++] = CFM;
	OutBuffer[OutSize++] = WGT_READ_WEIGHT;
	OutBuffer[OutSize++] = 1;	//data bytes

	OutBuffer[OutSize++] = Hwt_Weight;	
	(void)SerialComm_SendData();
	ActualMeasurement = NO_MEASUREMENT;				
}





#define TIMER_SEND_SPIROMETER_GRAPH_INTERVAL	30		//ms

UINT16 Spirometer_Fet;
UINT16 Spirometer_Fvc;
UINT16 Spirometer_Fev;
UINT16 Spirometer_Fev1;
UINT16 Spirometer_Peft;
UINT16 Spirometer_Fiv;
UINT16 Spirometer_Fef;
UINT16 Spirometer_Fif;
UINT16 Spirometer_Pef;

/* Spr */

void SprDiagnosticModeStartMeasurementReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = SPR_DIAGNOSTIC_MODE_START_MEASUREMENT;
		OutBuffer[OutSize++] = 1;	//data bytes
		
		if (Spr_StartMeasurement() == TRUE)
		{		
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}

		(void)SerialComm_SendData();
		ActualMeasurement = SPR_MEASUREMENT;
	}
}

void SprDiagnosticModeStopMeasurementReq(void)
{
	if (ActualMeasurement == SPR_MEASUREMENT)
	{
		Spr_AbortMeasurement();
		
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = SPR_ABORT_MEASUREMENT;
		OutBuffer[OutSize++] = 0;	//data bytes

		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
	}
}

void SprDiagnosticModeNewDataReadyInd(void)
{
	if (ActualMeasurement == SPR_MEASUREMENT)
	{
		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = SPR_DIAGNOSTIC_MODE_NEW_DATA_READY;	
		OutBuffer[OutSize++] = SPR_OUTBUFFER_SIZE;				//data bytes
		OutSize+=SPR_OUTBUFFER_SIZE;
		
		//data is alredy in the buffer
		(void)SerialComm_SendData();		
	}
}



/* Pox  */
void PoxStartMeasurementReq(void)
{	
	//execute command and send confirm
	OutBuffer[OutSize++] = CFM;
	OutBuffer[OutSize++] = POX_START_MEASUREMENT;
	OutBuffer[OutSize++] = 1;	//data bytes
	
	if (Pox_StartMeasurement() == TRUE)
	{
		OutBuffer[OutSize++] = ERROR_OK;	
	}
	else
	{
		OutBuffer[OutSize++] = ERROR_BUSY;
	}
	
	(void)SerialComm_SendData();
}



void PoxAbortMeasurementReq(void)
{
	//execute command and send confirm
	OutBuffer[OutSize++] = CFM;
	OutBuffer[OutSize++] = POX_ABORT_MEASUREMENT;
	OutBuffer[OutSize++] = 0;	//data bytes

	//execute command here
	Pox_AbortMeasurement();
	
	(void)SerialComm_SendData();
}



void PoxMeasurementCompleteOkInd(void)
{
	UINT8 Saturation = 98;
	UINT8 HeartRate = 60;

	OutBuffer[OutSize++] = IND;
	OutBuffer[OutSize++] = POX_MEASUREMENT_COMPLETE_OK;
	OutBuffer[OutSize++] = 2;				//data bytes
	
	OutBuffer[OutSize++] = Saturation;
	OutBuffer[OutSize++] = HeartRate;
		
	(void)SerialComm_SendData();
}

void PoxMeasurementErrorInd(void)
{
	OutBuffer[OutSize++] = IND;
	OutBuffer[OutSize++] = POX_MEASUREMENT_ERROR;
	OutBuffer[OutSize++] = 0;
	(void)SerialComm_SendData();
}



void PoxDiagnosticModeStartMeasurementReq(void)
{
	//execute command and send confirm
	OutBuffer[OutSize++] = CFM;
	OutBuffer[OutSize++] = POX_START_MEASUREMENT;
	OutBuffer[OutSize++] = 1;	//data bytes
	
	if (Pox_DiagnosticModeStartMeasurement() == TRUE)
	{
		OutBuffer[OutSize++] = ERROR_OK;	
	}
	else
	{
		OutBuffer[OutSize++] = ERROR_BUSY;
	}
	
	(void)SerialComm_SendData();
}

void PoxDiagnosticModeStopMeasurementReq(void)
{
	OutBuffer[OutSize++] = CFM;
	OutBuffer[OutSize++] = POX_ABORT_MEASUREMENT;
	OutBuffer[OutSize++] = 0;	//data bytes

	//execute command here
	Pox_DiagnosticModeStopMeasurement();
	
	(void)SerialComm_SendData();
}

void PoxNewDataReadyInd(void)
{
	static UINT16 PoxPacketId = 0;
	UINT8 i;
	//copy PoxBuffer to OutBuffer and send data
	OutBuffer[OutSize++] = IND;
	OutBuffer[OutSize++] = POX_DIAGNOSTIC_MODE_NEW_DATA_READY;
	OutBuffer[OutSize++] = POX_REAL_TIME_DATA_ARRAY_LENGTH + 4;	//packetId(2 bytes) + Pox data(64 bytes) + SpO2(1 bytes) + heart beat(1 bytes)
	
	OutBuffer[OutSize++] = (UINT8) (PoxPacketId >> 8);
	OutBuffer[OutSize++] = (UINT8) (PoxPacketId & 0x00FF);
	
	PoxPacketId++;
			
	for (i=0; i<POX_REAL_TIME_DATA_ARRAY_LENGTH; i++)
	{
		OutBuffer[OutSize++] = PoxGraph[i];
	}
	
	OutBuffer[OutSize++] = 98;	//SpO2
	OutBuffer[OutSize++] = 112;	//HR	
	
	(void)SerialComm_SendData();

}


void PoxDebugModeNewDataReadyInd(void)
{
	static UINT16 PoxPacketId = 0;
	UINT8 i;
	//copy PoxBuffer to OutBuffer and send data
	OutBuffer[OutSize++] = IND;
	OutBuffer[OutSize++] = POX_DEBUG_MODE_NEW_DATA_READY;
	OutBuffer[OutSize++] = POX_REAL_TIME_DATA_ARRAY_LENGTH + 5;	//packetId(2 bytes) + Pox data(64 bytes) + SpO2(1 bytes) + heart beat(1 bytes) + //HeartBeatDetected
	
	OutBuffer[OutSize++] = (UINT8) (PoxPacketId >> 8);
	OutBuffer[OutSize++] = (UINT8) (PoxPacketId & 0x00FF);
	
	PoxPacketId++;
			
	for (i=0; i<POX_REAL_TIME_DATA_ARRAY_LENGTH; i++)
	{
		OutBuffer[OutSize++] = PoxGraph[i];
	}

	OutBuffer[OutSize++] = Pox_SaturationValue;	//SpO2
	OutBuffer[OutSize++] = Pox_HeartRate;	//HR
	OutBuffer[OutSize++] = Pox_HeartBeatDetected;
			
	Pox_HeartBeatDetected = FALSE;	
	(void)SerialComm_SendData();
}


#endif



/*****************************************************************************
 * Local Functions
 *****************************************************************************/
/*****************************************************************************
 *
 *    @name     Init_Sys
 *
 *    @brief    This function Initializes the system
 *
 *    @param    None
 *
 *    @return   None
 *
 ****************************************************************************
 * Intializes the MCU, MCG, KBI, RTC modules
 ***************************************************************************/
static void Init_Sys(void)
{
    Mcu_Init(); /* initialize the MCU registers */
    MCG_Init(); /* initialize the MCG to generate 24MHz bus clock */
    //Kbi_Init(); /* initialize the KBI registers */
}

/*****************************************************************************
 * @name     MCU_Init
 *
 * @brief:   Initialization of the MCU.
 *
 * @param  : None
 *
 * @return : None
 *****************************************************************************
 * It will configure the MCU to disable STOP and COP Modules.
 * It also set the MCG configuration and bus clock frequency.
 ****************************************************************************/
static void Mcu_Init()
{


  /* Disable watchdog. */
    SOPT1 = 0x23; /* disable COP; enable bkgd, stop and wait mode */
 
    SOPT2 = 0x06; /* set right endianess for USB module */
    SOPT3 = SOPT3_SCI1PS_MASK;/* route TX1,RX1 to PTD6,PTD7 */

  /* Configure LED io pins to be outputs. 
   * EVB51JM128: LED to port mapping: 
   *  LED1  LED2 LED3 LED4 LED5 LED6 LED7 LED8
   *   |     |    |    |    |    |    |    |
   *   ^     ^    ^    ^    ^    ^    ^    ^
   *  PTE2  PTE3 PTF0 PTF1 PTF5 PTD2 PTC4 PTC6
   * DEMO51JM128: LED to port mapping: 
   *  LED1  LED2 LED3 LED4 LED5 LED6 LED7 LED8
   *   |     |    |    |    |    |    |    |
   *   ^     ^    ^    ^    ^    ^    ^    ^
   *  PTE2  PTE3 PTF0 PTF1 PTC2 PTC4 PTF5 PTD2
   */
  PTEDD= PTED_PTED2_MASK | PTED_PTED3_MASK;
  PTFDD= PTFD_PTFD0_MASK | PTFD_PTFD1_MASK | PTFD_PTFD5_MASK;
  PTCDD= PTCD_PTCD4_MASK 
 #ifdef EVB
          | PTCD_PTCD6_MASK
 #else
          | PTCD_PTCD2_MASK          
 #endif          
          ;
          
  PTDDD= PTDD_PTDD2_MASK;
  
  /* Enable internal pull-ups on port E pins to get switches working.
   * EVB51JM128: Button-to-port mapping:
   *  SW1   SW2   SW3   SW4
   *   |     |     |     |
   *   ^     ^     ^     ^
   *  PTG1  PTG2  PTG3  PTD5
   * DEMO51JM128: Button-to-port mapping:
   *  SW1   SW2   SW3   SW4
   *   |     |     |     |
   *   ^     ^     ^     ^
   *  PTG0  PTG1  PTG2  PTG3
   */
  PTGPE= PTGPE_PTGPE1_MASK | PTGPE_PTGPE2_MASK | PTGPE_PTGPE3_MASK
  #ifndef EVB
    | PTGPE_PTGPE0_MASK  
  #endif
   ;
  PTDPE= PTDPE_PTDPE5_MASK ;
  
  IRQSC = IRQSC_IRQIE_MASK | IRQSC_IRQPE_MASK; /*enable the IRQ interrupt for SW4 */    

}

/*****************************************************************************
 * @name     MCG_Init
 *
 * @brief:   Initialization of the Multiple Clock Generator.
 *
 * @param  : None
 *
 * @return : None
 *****************************************************************************
 * Provides clocking options for the device, including a phase-locked
 * loop(PLL) and frequency-locked loop (FLL) for multiplying slower reference
 * clock sources
 ****************************************************************************/
static void MCG_Init()
{

 /* Assume 12MHz external clock source connected. */
  
  /* In order to use the USB we need to enter PEE mode and MCGOUT set to 48 MHz. 
     Out of reset MCG is in FEI mode. */
  
  
  
  
  //------FBE MODE------  
  /* Assume 16MHz external clock source connected. */
  // RANGE = 1; HGO = 1; ERCLKEN=1; EREFS = 1; BDIV = 000
  
  MCGC2 = 0x36;
  
  
  
  // DIV32 =1
  MCGC3 = 0x10;
  
  // CLKS = 10; RDIV = 100; IREFS = 0;
  MCGC1 = 0xA0; 


  // Wait for Reference Status bit to update
  while (MCGSC_IREFST)
  {
  }
  // Wait for clock status bits to update 
  while (MCGSC_CLKST != 0b10) 
  {
  }

//------PBE MODE------ 

  // PLLS =1; DIV32 = 1; VDIV = 1001 
  MCGC3 = 0x5c;
  
  // wait for PLL status bit to update
  while (!MCGSC_PLLST) 
  {
  }
  // Wait for LOCK bit to set 
  while (!MCGSC_LOCK) 
  {
  }
  
//------PEE MODE------   

  // CLKS = 00; RDIV = 100; IREFS = 0
  MCGC1 = 0x20;

// Wait for clock status bits to update 
  while (MCGSC_CLKST != 0b11) 
  {
  }
}

